//	Maze4DGestures.swift
//
//	© 2025 by Jeff Weeks
//	See TermsOfUse.txt

import SwiftUI


let maxTranslationalSpeed: Double = 2.0	//	radians/second
let minTranslationalSpeed: Double = 0.1	//	radians/second, to suppress any small unintended motions
let minFrameθ: Double = minTranslationalSpeed * gFramePeriod
let maxFrameθ: Double = maxTranslationalSpeed * gFramePeriod

let π = Double.pi


// MARK: -
// MARK: Drag gesture

enum Maze4DDragPurpose {
	case unknown
	case toMoveSlider
	case toRotateMaze
}

struct Maze4DDragState {

	var purpose: Maze4DDragPurpose = .unknown

	//	When the slider is sitting at a node and user wants to move it,
	//	we'll need to decide which of the available tubes the user
	//	is trying to move it along.  To reliably decide which tube,
	//	we'll wait until the user's finger has moved some nontrivial distance,
	//	so we'll have a good sense of the intended direction.
	var previousSliderTouchIntrinsic: SIMD2<Double> = .zero

	//	dragToMoveSliderEnded() will need to know dragToMoveSliderChanged()'s
	//	last computed value of the ratio
	//
	//		distance the slider moved (in edge coordinates 0.0 … 1.0)
	//		----------------------------------------------------------
	//		distance the touch point moved (in instrinsic coordinates)
	//
	//	This ratio takes into account both
	//	- the edge's angle relative to the display and
	//	- the drag-path's angle relative to the edge's projection onto the display.
	//
	var edgeDistancePerDragDistance: Double? = nil

	//	While rotating the figure, keep track
	//	of the previous drag point on the unit sphere.
	var previousPoint: SIMD3<Double>? = nil	//	on unit sphere
}

func maze4DDragGesture(
	modelData: Maze4DModel,
	viewSize: CGSize,
	inertia: Bool,
	dragState: GestureState<Maze4DDragState>
) -> some Gesture {

	//	SwiftUI will clear the dragState before calling onEnded,
	//	so let's keep separate copies of the purpose and
	//	the edgeDistancePerDragDistance for onEnded to use.
	var thePurposeCopy
			= dragState.wrappedValue.purpose
	var theEdgeDistancePerDragDistanceCopy
			= dragState.wrappedValue.edgeDistancePerDragDistance

	let theDragGesture = DragGesture()
	.updating(dragState) { value, theDragState, transaction in
	
		switch theDragState.purpose {
		
		case .unknown:
			
			dragBegan(
				dragState: &theDragState,
				modelData: modelData,
				startLocation: value.startLocation,
				viewSize: viewSize)
			
		case .toMoveSlider:
		
			dragToMoveSliderChanged(
				dragState: &theDragState,
				modelData: modelData,
				location: value.location,
				viewSize: viewSize)
			
		case .toRotateMaze:
			
			dragToRotateMazeChanged(
				dragState: &theDragState,
				modelData: modelData,
				startLocation: value.startLocation,
				location: value.location,
				viewSize: viewSize)
		}
		
		thePurposeCopy = theDragState.purpose
		theEdgeDistancePerDragDistanceCopy = theDragState.edgeDistancePerDragDistance
	}
	.onEnded() { value in

		//	Caution:  This onEnded callback gets called
		//	when the gesture ends normally, but not when,
		//	say, a DragGesture gets interrupted when
		//	the user places a second finger on the display.
	
		switch thePurposeCopy {
		
		case .unknown:
			break	//	should never occur
			
		case .toMoveSlider:
		
			dragToMoveSliderEnded(
				modelData: modelData,
				velocity: value.velocity,
				viewSize: viewSize,
				edgeDistancePerDragDistance: theEdgeDistancePerDragDistanceCopy)
			
		case .toRotateMaze:
			
			dragToRotateMazeEnded(
				modelData: modelData,
				location: value.location,
				velocity: value.velocity,
				viewSize: viewSize,
				inertia: inertia)
		}
		
		thePurposeCopy = .unknown
	}
	
	return theDragGesture
}

func mapTouchPointToIntrinsicCoordinates(
	touchPoint: CGPoint,	//	0 ≤ touchPoint.x|y ≤ viewSize.width|height
	viewSize: CGSize
) -> SIMD2<Double> {

	precondition(
		viewSize.width > 0.0 && viewSize.height > 0.0,
		"mapTouchPointToIntrinsicCoordinates() received viewSize of non-positive width or height")

	//	Shift the coordinates to place the origin at the center of the view.
	var x = touchPoint.x - 0.5 * viewSize.width
	var y = touchPoint.y - 0.5 * viewSize.height

	//	Flip from iOS's Y-down coordinates
	//	to Metal's Y-up coordinates.
	y = -y

	//	Convert the coordinates to intrinsic units (IU).
	let theIntrinsicUnitsPerPixelOrPoint = intrinsicUnitsPerPixelOrPoint(
												viewWidth: viewSize.width,
												viewHeight: viewSize.height)
	x *= theIntrinsicUnitsPerPixelOrPoint
	y *= theIntrinsicUnitsPerPixelOrPoint
	
	return SIMD2<Double>(x, y)
}

func mapTouchVelocityToIntrinsicUnitsPerSecond(
	touchVelocity: CGSize,	//	in points per second
	viewSize: CGSize
) -> SIMD2<Double> {

	precondition(
		viewSize.width > 0.0 && viewSize.height > 0.0,
		"mapTouchVelocityToIntrinsicCoordinates() received viewSize of non-positive width or height")

	//	Flip from iOS's Y-down coordinates
	//	to Metal's Y-up coordinates.
	var Δx =  touchVelocity.width
	var Δy = -touchVelocity.height

	//	Convert the components to intrinsic units (IU) per second.
	let theIntrinsicUnitsPerPixelOrPoint = intrinsicUnitsPerPixelOrPoint(
												viewWidth: viewSize.width,
												viewHeight: viewSize.height)
	Δx *= theIntrinsicUnitsPerPixelOrPoint
	Δy *= theIntrinsicUnitsPerPixelOrPoint
	
	return SIMD2<Double>(Δx, Δy)
}

func dragBegan(
	dragState: inout Maze4DDragState,
	modelData: Maze4DModel,
	startLocation: CGPoint,
	viewSize: CGSize
) {

	let theIntrinsicStartPoint = mapTouchPointToIntrinsicCoordinates(
									touchPoint: startLocation,
									viewSize: viewSize)

	dragState.purpose = findDragPurpose(
							modelData: modelData,
							touchPointIntrinsic: theIntrinsicStartPoint)

	//	We'll need to use previousSliderTouchIntrinsic
	//	iff the user is moving the slider away from a node.
	dragState.previousSliderTouchIntrinsic = theIntrinsicStartPoint
	
	//	Stop the automatic rotation
	//	no matter what the drag's purpose is.
	modelData.itsIncrement = nil
	
	//	If the drag's purpose is toMoveSlider,
	//	stop any pre-existing coasting.
	if dragState.purpose == .toMoveSlider {
		modelData.itsMaze?.sliderCoastingSpeed = nil
	}
}

func findDragPurpose(
	modelData: Maze4DModel,
	touchPointIntrinsic: SIMD2<Double>
) -> Maze4DDragPurpose {

	guard let theMaze = modelData.itsMaze else {
		return .toRotateMaze
	}

	//	Ignore slider hits after the game is over.
	if modelData.itsGameIsOver {
		return .toRotateMaze
	}
	
	let theHitTestRay = makeHitTestRay(
							touchPointIntrinsic: touchPointIntrinsic,
							orientation: modelData.itsOrientation,
							n: theMaze.difficultyLevel.n)
	
	let (theSliderCenter, _) = sliderOffsetCenterAndColor(
								sliderPosition: theMaze.sliderPosition,
								difficulty: theMaze.difficultyLevel,
								shearFactor: modelData.its4DShearFactor)

	//	Accept touches in a "halo" that's bigger than the slider itself.
	//
	//		The factor of (n - 1) makes the halo a constant size
	//		in world coordinates instead of a constant size
	//		in maze coordinates.
	//
	let theHaloRadius = 2.0
					  * Double( theMaze.difficultyLevel.n - 1 )
					  * sliderRadius

	if rayHitsSlider(
		ray: theHitTestRay,
		sliderCenter: theSliderCenter,
		haloRadius: theHaloRadius) {

		return .toMoveSlider
		
	} else {
	
		return .toRotateMaze
	}
}

func dragToMoveSliderChanged(
	dragState: inout Maze4DDragState,
	modelData: Maze4DModel,
	location: CGPoint,
	viewSize: CGSize
) {

	guard let theMaze = modelData.itsMaze else {
		assertionFailure("Internal error:  itsMaze is nil in dragToMoveSliderChanged()")
		return
	}
	
	//	As soon as the user touches the slider,
	//	stop the blinking and flash the goal instead.
	if modelData.itsSliderBlinkStartTime != nil {
	
		modelData.itsSliderBlinkStartTime = nil
		modelData.itsGoalFlashStartTime = CFAbsoluteTimeGetCurrent()
	}

	let theIntrinsicTouchPoint = mapTouchPointToIntrinsicCoordinates(
									touchPoint: location,
									viewSize: viewSize)
	
	switch theMaze.sliderPosition {
	
	case .atNode(nodeIndex: let nodeIndex):
	
		dragSliderFromNode(
			modelData: modelData,
			touchPointIntrinsic: theIntrinsicTouchPoint,
			previousSliderTouchIntrinsic: dragState.previousSliderTouchIntrinsic,
			nodeIndex: nodeIndex)
		
	case .onOutboundEdge(
			baseNodeIndex: let baseNodeIndex,
			direction: let direction,
			distance: let distance):
		
		dragSliderAlongEdge(
			dragState: &dragState,
			modelData: modelData,
			touchPointIntrinsic: theIntrinsicTouchPoint,
			baseNodeIndex: baseNodeIndex,
			direction: direction,
			distance: distance)
	}
}

func dragSliderFromNode(
	modelData: Maze4DModel,
	touchPointIntrinsic: SIMD2<Double>,
	previousSliderTouchIntrinsic: SIMD2<Double>,
	nodeIndex: MazeNodeIndex
) {

	guard let theMaze = modelData.itsMaze else {
		assertionFailure("Internal error:  itsMaze is nil in dragToMoveSliderChanged()")
		return
	}
	let theOrientation = modelData.itsOrientation
	let theShearFactor = modelData.its4DShearFactor
	
	//	We must decide which tube the user wants to slide the slider along
	//	before we can start sliding it.
	//
	//	One or two onChanged() events might not be enough to select a tube:
	//	if the user moves the his/her finger very slowly,
	//	the initial (Δx, Δy) may be for only 1 pixel in some direction,
	//	for example (+1,0) or (0,-1).  Such a displacement might align
	//	more closely to an undesired tube than to the one the user
	//	is trying to move the slider along.  Better to accumulate
	//	a larger displacement, like (+3,-5), before selecting the tube.
	//	As a practical matter, let's accumulate a finger motion
	//	of at least a few percent of the bounding box width before proceeding.
	
	let theTouchMotion = touchPointIntrinsic - previousSliderTouchIntrinsic
	if length(theTouchMotion) < 0.03125 * (2.0 * boxSize) {
	
		//	Wait for the user to move his/her finger a bit further
		//	before we select a tube.
		return
	}
	
	let theOriginToDisplayDistance = boxSize
	let theDisplayToObserverDistance = perspectiveFactor * theOriginToDisplayDistance
	let theOriginToObserverDistance = theOriginToDisplayDistance
									+ theDisplayToObserverDistance

	//	In world coordinates, an observer at (0, 0, -OrgToObs)
	//	projects the scene onto the plane z = -OrgToDsp via the function
	//
	//		               DspToObs        DspToObs
	//		p(x,y,z) = ( ------------ x, ------------ y, -OrgToDsp )
	//		             z + OrgToObs    z + OrgToObs
	//
	//	The partial derivatives
	//
	//		∂p        DspToObs
	//		-- = (  ------------,            0,           0 )
	//		∂x      z + OrgToObs
	//
	//		∂p                           DspToObs
	//		-- = (        0,           ------------,      0 )
	//		∂y                         z + OrgToObs
	//
	//		∂p       -DspToObs          -DspToObs
	//		-- = ( --------------- x, --------------- y,  0 )
	//		∂z     (z + OrgToObs)²    (z + OrgToObs)²
	//
	//	tell how the projected point responds to movements
	//	of the original point (x,y,z).

	let (theSliderPosition, _) = sliderOffsetCenterAndColor(
									sliderPosition: theMaze.sliderPosition,
									difficulty: theMaze.difficultyLevel,
									shearFactor: theShearFactor)
	
	//	Rescale theSliderPosition from [0, n-1] coordinates to [-1, +1] coordinates.
	let theRescaledPosition = SIMD3<Double>(-1.0, -1.0, -1.0)
							+ 2.0 * theSliderPosition / Double(theMaze.difficultyLevel.n - 1)
							
	//	Rotate theRescaledPosition to world coordinates, as seen by the user.
	let theRotatedPosition = theOrientation.act(theRescaledPosition)
	
	let x = theRotatedPosition[0]
	let y = theRotatedPosition[1]
	let z = theRotatedPosition[2]
	
	let theFactorA	= theDisplayToObserverDistance / (z + theOriginToObserverDistance)
	let theFactorB	=         -theFactorA          / (z + theOriginToObserverDistance)

	let theProjectionDerivative = simd_double3x2(
		SIMD2<Double>(  theFactorA,         0.0     ),
		SIMD2<Double>(      0.0,        theFactorA  ),
		SIMD2<Double>(theFactorB * x, theFactorB * y))

	//	Apply the derivative to project the basis vectors
	//	onto the plane of the display.
	//
	let r3 = sqrt(1.0 / 3.0)
	let theRawBasisVectors = [
		SIMD3<Double>(1.0, 0.0, 0.0),	//	x direction
		SIMD3<Double>(0.0, 1.0, 0.0),	//	y direction
		SIMD3<Double>(0.0, 0.0, 1.0),	//	z direction
		SIMD3<Double>( r3,  r3,  r3)	//	w direction represented as axis in 3-space
	]
	let theRotatedBasisVectors = theRawBasisVectors.map() { v in
		theOrientation.act(v)
	}
	let theProjectedBasisVectors = theRotatedBasisVectors.map() { v in
		theProjectionDerivative * v		//	right-to-left action
	}

	//	Normalize each of theProjectedBasisVectors to unit length, if possible.
	//	Avoid the built-in normalize() function, whose documentation
	//	doesn't say what happens if you pass it a zero vector.
	//
	let theNormalizedBasisVectors = theProjectedBasisVectors.map() { v in
		let theLength = length(v)
		return ( theLength > 0.0 ? v / theLength : SIMD2<Double>.zero)
	}

	//	Which basis vector best matches theMotion?
	let theEdges = theMaze.edges[nodeIndex[0]][nodeIndex[1]][nodeIndex[2]][nodeIndex[3]]
	var theBestMotionComponent = 0.0
	var theBestAxis = 0
	var theBestIsOutbound = false
	for i in 0...3 {
	
		//	Project theTouchMotion onto theNormalizedBasisVectors[i].
		//	The latter is a unit vector (or zero) so the dot product
		//	gives the length of theTouchMotion's projection.
		//
		let theMotionComponent = dot(theTouchMotion, theNormalizedBasisVectors[i])
		
		if theEdges.outbound[i]
		&& +theMotionComponent > theBestMotionComponent {
		
			theBestMotionComponent	= +theMotionComponent
			theBestAxis				= i
			theBestIsOutbound		= true
		}
		
		if theEdges.inbound[i]
		&& -theMotionComponent > theBestMotionComponent {
		
			theBestMotionComponent	= -theMotionComponent
			theBestAxis				= i
			theBestIsOutbound		= false
		}
	}

	//	Did we find an acceptable edge to move along?
	if theBestMotionComponent > 0.0 {
	
		var theBaseNodeIndex = nodeIndex
		var theDistance: Double
		if theBestIsOutbound {
		
			theDistance = 0.0
			
		} else {

			precondition(
				theBaseNodeIndex[theBestAxis] > 0,
				"Internal error:  impossible inbound edge in dragSliderFromNode()")
			theBaseNodeIndex[theBestAxis] -= 1
		
			theDistance = 1.0
		}
	
		let theNewSliderPosition = SliderPosition.onOutboundEdge(
				baseNodeIndex: theBaseNodeIndex,
				direction: theBestAxis,
				distance: theDistance)
			
		//	Caution:  theMaze is a value struct, so we must
		//	write theNewSliderPosition directly into the modelData,
		//	which gets passed by reference.
		modelData.itsMaze?.sliderPosition = theNewSliderPosition
	}
}

func dragSliderAlongEdge(
	dragState: inout Maze4DDragState,
	modelData: Maze4DModel,	//	modelData is passed by reference, so we can modify the slider
	touchPointIntrinsic: SIMD2<Double>,
	baseNodeIndex: MazeNodeIndex,
	direction: Int,		//	which axis the slider is displaced along ∈ {0, 1, 2, 3}
	distance: Double	//	how far the slider is from the base node
						//		0.0 ≤ distance ≤ 1.0
) {

	guard let theMaze = modelData.itsMaze else {
		assertionFailure("Internal error:  itsMaze is nil in dragSliderAlongEdge()")
		return
	}
	let theOrientation = modelData.itsOrientation
	let theShearFactor = modelData.its4DShearFactor
	let n = theMaze.difficultyLevel.n

	let theTouchMotion = touchPointIntrinsic - dragState.previousSliderTouchIntrinsic
	dragState.previousSliderTouchIntrinsic = touchPointIntrinsic

	var farNodeIndex = baseNodeIndex
	farNodeIndex[direction] += 1

	let theEndpointIndices = [baseNodeIndex, farNodeIndex]

	let theShearedEndpoints = theEndpointIndices.map() { nodeIndex in
	
		let (theShearedPosition, _, _) = shearedPosition(
											position4D: SIMD4<Double>(nodeIndex),
											difficulty: theMaze.difficultyLevel,
											shearFactor: theShearFactor)

		return theShearedPosition
	}
	
	//	Rescale theShearedEndpoints from [0, n-1] coordinates to [-1, +1] coordinates.
	let theRescaledEndpoints = theShearedEndpoints.map() { endpoint in
	
		SIMD3<Double>(-1.0, -1.0, -1.0) + 2.0 * endpoint / Double(n - 1)
	}
							
	//	Rotate theRescaledEndpoints to world coordinates, as seen by the user.
	let theRotatedEndpoints = theRescaledEndpoints.map() { endpoint in
		
		theOrientation.act(endpoint)
	}

	//	Project theRotatedEndpoints onto the plane of the display.
	let theProjectedEndpoints = theRotatedEndpoints.map() { endpoint -> SIMD2<Double> in
	
		let x = endpoint[0]
		let y = endpoint[1]
		let z = endpoint[2] + (perspectiveFactor + 1.0) * boxSize	//	maps to camera coordinates
		
		if z <= 0.0 {
			assertionFailure("Internal error:  endpoint sits behind camera in dragSliderAlongEdge()")
			return .zero
		}
		
		let theProjectionFactor = perspectiveFactor * boxSize / z
		
		return SIMD2<Double>(
				x * theProjectionFactor,
				y * theProjectionFactor)
	}

	//	theProjectedEdge lies in the plane of the display.
	let theProjectedEdge = theProjectedEndpoints[1] - theProjectedEndpoints[0]
	let theProjectedEdgeLength = length(theProjectedEdge)
	if theProjectedEdgeLength < 0.001 {
		//	The user is seeing the edge end-on,
		//	and therefore cannot slide the slider along it.
		//	The player will need to rotate the maze
		//	in order to be able to slide the slider.
		return
	}
	
	let theUnitProjectedEdge = theProjectedEdge / theProjectedEdgeLength
	
	//	Compute theTouchMotion's component parallel to theUnitProjectedEdge.
	let theParallelComponent = dot(theTouchMotion, theUnitProjectedEdge)
	
	//	Express theParallelComponent as a fraction of theProjectedEdgeLength.
	let theFractionalMotion = theParallelComponent / theProjectedEdgeLength

	//	We could explicitly invert the projection to implement
	//	a more sophisticated mouse-tracking algorithm, but for now
	//	let's simply move the slider along the edge through theFractionalMotion
	//	and see how that feels when solving a maze.
	//	(Answer:  In practice it feels great!  So let's leave this code
	//	the way it is, and not bother with a more precise algorithm.)
	let theNewSliderDistance = distance + theFractionalMotion

	//	If the slider has reached or passed an endpoint,
	//	snap it to that endpoint and set itsStatus to SliderAtNode.
	let theNewSliderPosition: SliderPosition
	if theNewSliderDistance <= 0.0 {
		theNewSliderPosition = .atNode(nodeIndex: baseNodeIndex)
	} else if theNewSliderDistance > 1.0 {
		theNewSliderPosition = .atNode(nodeIndex: farNodeIndex)
	} else {
		theNewSliderPosition = .onOutboundEdge(
									baseNodeIndex: baseNodeIndex,
									direction: direction,
									distance: theNewSliderDistance)
	}

	switch theNewSliderPosition {
	case .atNode(_):
		dragState.edgeDistancePerDragDistance = nil
		
	case .onOutboundEdge(_, _, _):
		dragState.edgeDistancePerDragDistance = theFractionalMotion
											  / length(theTouchMotion)
	}

	//	Caution:  theMaze is a value struct, so we must
	//	write theSnappedSliderPosition directly into the modelData,
	//	which gets passed by reference.
	modelData.itsMaze?.sliderPosition = theNewSliderPosition
}

func dragToMoveSliderEnded(
	modelData: Maze4DModel,	//	modelData is passed by reference, so we can modify the slider
	velocity: CGSize,
	viewSize: CGSize,
	edgeDistancePerDragDistance: Double?
) {

	snapSliderToEndpoint(modelData: modelData)
	
	if !modelData.itsGameIsOver
	&& sliderHasReachedGoal(maze: modelData.itsMaze) {
	
		modelData.itsGameIsOver = true
	}

	//	If the slider was moving along an edge when the drag ended,
	//	set an appropriate coasting speed.
	//
	if let theSliderPosition = modelData.itsMaze?.sliderPosition {

		switch theSliderPosition {
		
		case .atNode(nodeIndex: _):
			
			modelData.itsMaze?.sliderCoastingSpeed = nil
			
		case .onOutboundEdge(baseNodeIndex: _, direction: _, distance: _):
			
			let theVelocityInIntrinsicUnitsPerSecond
				= mapTouchVelocityToIntrinsicUnitsPerSecond(
					touchVelocity: velocity,
					viewSize: viewSize)

			let theDragSpeed = length(theVelocityInIntrinsicUnitsPerSecond)
			
			if let theEdgeDistancePerDragDistance = edgeDistancePerDragDistance {
			
				let theSliderSpeed = theEdgeDistancePerDragDistance * theDragSpeed
				
				modelData.itsMaze?.sliderCoastingSpeed = theSliderSpeed
			
			} else {
			
				modelData.itsMaze?.sliderCoastingSpeed = nil
			}
		}
	}
}

func snapSliderToEndpoint(
	modelData: Maze4DModel	//	modelData is passed by reference, so we can modify the slider
) {

	guard let theMaze = modelData.itsMaze else {
		assertionFailure("Internal error:  itsMaze is nil in snapSliderToEndpoint()")
		return
	}
	
	let theSliderPosition = theMaze.sliderPosition

	//	If the slider is close to a node, snap to it exactly.
	let theSnapToNodeTolerance = 0.125
	let theSnappedSliderPosition: SliderPosition
	switch theSliderPosition {
	
	case .atNode(nodeIndex: _):
		
		theSnappedSliderPosition = theSliderPosition	//	no change
		
	case .onOutboundEdge(
			baseNodeIndex: let baseNodeIndex,
			direction: let direction,
			distance: let distance):

		if distance < 0.0 + theSnapToNodeTolerance {
		
			theSnappedSliderPosition = .atNode(nodeIndex: baseNodeIndex)
			
		} else
		if distance > 1.0 - theSnapToNodeTolerance {

			var farNodeIndex = baseNodeIndex
			farNodeIndex[direction] += 1

			theSnappedSliderPosition = .atNode(nodeIndex: farNodeIndex)

		} else {
		
			theSnappedSliderPosition = theSliderPosition	//	no change
		}
	}

	//	Caution:  theMaze is a value struct, so we must
	//	write theSnappedSliderPosition directly into the modelData,
	//	which gets passed by reference.
	modelData.itsMaze?.sliderPosition = theSnappedSliderPosition
}

func dragToRotateMazeChanged(
	dragState: inout Maze4DDragState,
	modelData: Maze4DModel,
	startLocation: CGPoint,
	location: CGPoint,
	viewSize: CGSize
) {

	let theIntrinsicStartPoint = mapTouchPointToIntrinsicCoordinates(
									touchPoint: startLocation,
									viewSize: viewSize)
	let theIntrinsicTouchPoint = mapTouchPointToIntrinsicCoordinates(
									touchPoint: location,
									viewSize: viewSize)

	//	On the first call to dragToRotateMazeChanged()
	//	during a given drag, the previousPoint will be nil
	//	and we'll use the drag's startLocation instead.
	//	Thereafter we'll store the point from one call
	//	for use in the next.
	//
	let p₀ = dragState.previousPoint ??
		mapToUnitSphere(touchPointIntrinsic: theIntrinsicStartPoint)

	let p₁ = mapToUnitSphere(touchPointIntrinsic: theIntrinsicTouchPoint)

	let (theAxis, θ) = parallelTransportAxisAndAngle(p₀: p₀, p₁: p₁)
	let theIncrement = simd_quatd(angle: θ, axis: theAxis)

	modelData.itsOrientation
		= simd_normalize( theIncrement * modelData.itsOrientation )

	//	Update thePreviousPoint for use in the next call to updating().
	dragState.previousPoint = p₁
}

func dragToRotateMazeEnded(
	modelData: Maze4DModel,
	location: CGPoint,
	velocity: CGSize,
	viewSize: CGSize,
	inertia: Bool
) {

	//	If itsOriention is almost axis aligned, snap to perfect alignment.
	if let theAxisAlignedOrientation
		= snapOrientationToAxes(modelData.itsOrientation) {
	
		modelData.itsOrientation = theAxisAlignedOrientation
		modelData.itsIncrement = nil
		
	} else {

		if inertia {
		
			let theTouchPoint₀ = location
			let theTouchPoint₁ = CGPoint(
				x: location.x + gFramePeriod * velocity.width,
				y: location.y + gFramePeriod * velocity.height)

			let theIntrinsicTouchPoint₀ = mapTouchPointToIntrinsicCoordinates(
											touchPoint: theTouchPoint₀,
											viewSize: viewSize)
			let theIntrinsicTouchPoint₁ = mapTouchPointToIntrinsicCoordinates(
											touchPoint: theTouchPoint₁,
											viewSize: viewSize)

			let p₀ = mapToUnitSphere(touchPointIntrinsic: theIntrinsicTouchPoint₀)
			let p₁ = mapToUnitSphere(touchPointIntrinsic: theIntrinsicTouchPoint₁)

			let (theAxis, θ) = parallelTransportAxisAndAngle(p₀: p₀, p₁: p₁)

			let theClampedθ = min(θ, maxFrameθ)

			//	If the user tries to stop the motion at the end of a drag,
			//	but leaves some some residual motion ( θ < minFrameθ ),
			//	set theIncrement to nil to stop the motion entirely.
			//
			let theIncrement = theClampedθ > minFrameθ ?
								simd_quatd(angle: theClampedθ, axis: theAxis) :
								nil
			
			modelData.itsIncrement = theIncrement

		} else { // inertia == false

			//	Redundant but makes our intention clear
			modelData.itsIncrement = nil
		}
	}

	if gGetScreenshotOrientations {
		print(modelData.itsOrientation)
		modelData.itsIncrement = nil	//	Suppress inertia
	}
}

func mapToUnitSphere(
	touchPointIntrinsic: SIMD2<Double>
) -> SIMD3<Double> {

	//	We'll interpret the touchPointIntrinsic relative
	//	to a sphere of radius boxSize, even though we'll
	//	ultimately return a unit vector.
	
	let x = touchPointIntrinsic[0] / boxSize
	let y = touchPointIntrinsic[1] / boxSize
	let r = sqrt( x*x + y*y )
	
	//	Use an orthogonal projection.  It's simpler than
	//	the perspective projection that the renderer
	//	uses to render the maze, yet feels just as natural.

	let p = ( r < 1.0 ?
		SIMD3<Double>(x, y, -sqrt(1.0 - r*r)) :	//	in southern hemisphere
		SIMD3<Double>(x/r, y/r, 0.0) )			//	on equator
		
	return p
}

func parallelTransportAxisAndAngle(
	p₀: SIMD3<Double>,	//	unit vector
	p₁: SIMD3<Double>	//	unit vector
) -> (SIMD3<Double>, Double)	//	(the axis, the angle) that parallel transports p₀ to p₁.
								//	The axis has unit length.
								//	The angle is always non-negative.
{
	//	Take a cross product to get the axis of rotation.

	let theCrossProduct = SIMD3<Double>(
							p₀.y * p₁.z  -  p₀.z * p₁.y,
							p₀.z * p₁.x  -  p₀.x * p₁.z,
							p₀.x * p₁.y  -  p₀.y * p₁.x )

	let theCrossProductLength = sqrt( theCrossProduct.x * theCrossProduct.x
									+ theCrossProduct.y * theCrossProduct.y
									+ theCrossProduct.z * theCrossProduct.z )
	
	let theAxis: SIMD3<Double>
	let θ: Double
	if theCrossProductLength > 0.0 {
	
		//	Normalize theCrossProduct to unit length
		//	to get a normalized axis of rotation.
		theAxis = theCrossProduct / theCrossProductLength

		//	p₀ and p₁ are both unit vectors, so
		//
		//		theCrossProductLength = |p₀|·|p₁|·sin(θ)
		//							  = sin(θ)
		//
		//		Note:  Using theCosine = p₀·p₁
		//		could be less numerically precise.
		//
		let theSine = theCrossProductLength
		let theSafeSine = min(theSine, 1.0)	//	guard against theSine = 1.0000000000001
		θ = asin(theSafeSine)

	} else {	//	theCrossProductLength = 0.0
	
		//	p₀ and p₁ are equal (or collinear) and the motion is the identity.
		//	We can pick an arbitrary axis and report zero distance.
		//
		//		Note #1:  The touch input values are discrete,
		//		so we shouldn't have to worry about including
		//		any sort of tolerance here.
		//
		//		Note #2:  We're unlikley to ever receive p₁ = -p₀.
		//
		theAxis = SIMD3<Double>(1.0, 0.0, 0.0)
		θ = 0.0
	}
	
	return (theAxis, θ)
}


// MARK: -
// MARK: Rotation gesture

func maze4DRotateGesture(
	modelData: Maze4DModel,
	previousAngle: GestureState<Double>
) -> some Gesture {

	//	When running on macOS (as a "designed for iPadOS" app)
	//	rotations will be recognized iff
	//
	//		Settings > Trackpad > Scroll & Zoom > Rotate
	//
	//	is enabled.  Fortunately that seems to be the default setting.

	let theRotateGesture = RotateGesture(minimumAngleDelta: .zero)
	.updating(previousAngle) { value, thePreviousAngle, transaction in

		//	Suppress the usual per-frame increment
		//	while the user is manually rotating the maze.
		modelData.itsIncrement = nil

		let theNewAngle = value.rotation.radians
		
		//	RotateGesture() sometimes returns theNewAngle = NaN. Ouch!
		if theNewAngle.isNaN {
			return
		}

		var Δθ = theNewAngle - thePreviousAngle

		//	Avoid discontinuous jumps by 2π ± ε
		if Δθ > π { Δθ -= 2.0 * π }
		if Δθ < π { Δθ += 2.0 * π }

		let theIncrement = simd_quatd(angle: Δθ, axis: SIMD3<Double>(0.0, 0.0, -1.0))
		modelData.itsOrientation = simd_normalize(theIncrement * modelData.itsOrientation)

		//	Update thePreviousAngle for next time.
		thePreviousAngle = theNewAngle
	}
	.onEnded() { _ in

		//	Trying to decide whether the user wants the maze
		//	to keep rotating or not at the end of the gesture is trickier
		//	than it seems. So let's just stop rotating, no matter what.
		//	The 2-finger rotation is an awkward gesture for the user
		//	to perform in any case. The 1-finger rotation -- with
		//	the user's finger near the edge of the display -- is
		//	an easier way to rotate the maze about an axis
		//	orthogonal to the display.
		modelData.itsIncrement = nil	//	redundant but clear

		//	If itsOriention is almost axis aligned, snap to perfect alignment.
		if let theAxisAlignedOrientation
			= snapOrientationToAxes(modelData.itsOrientation) {
		
			modelData.itsOrientation = theAxisAlignedOrientation
		}
	}
	
	return theRotateGesture
}


// MARK: -
// MARK: Coordinates

func snapOrientationToAxes(
	_ orientation: simd_quatd
) -> simd_quatd? {	//	returns nil if orientation isn't already close to axis-aligned

	//	The axis-aligned orientations of a cube correspond
	//	exactly to the elements of the group Isom(cube).
	//	Because we record the cube's orientation as a quaternion,
	//	we must list the elements of the corresponding "binary" group
	//	which maps 2-to-1 onto Isom(cube).
	//	The group Isom(cube) == Isom(octahedron) is most commonly
	//	called the octahedral group, and its 2-fold cover
	//	is the binary octahedral group.
	
	let rhf = 0.70710678118654752440	//	√½	(rhf = Root of one HalF)

	let theAxisAlignedOrientations: [simd_quatd] = [

		//	identity
		simd_quatd(ix: 0.0, iy: 0.0, iz: 0.0, r: 1.0), simd_quatd(ix: 0.0, iy: 0.0, iz: 0.0, r:-1.0),
		
		//	2-fold rotations about face centers
		simd_quatd(ix: 1.0, iy: 0.0, iz: 0.0, r: 0.0), simd_quatd(ix:-1.0, iy: 0.0, iz: 0.0, r: 0.0),
		simd_quatd(ix: 0.0, iy: 1.0, iz: 0.0, r: 0.0), simd_quatd(ix: 0.0, iy:-1.0, iz: 0.0, r: 0.0),
		simd_quatd(ix: 0.0, iy: 0.0, iz: 1.0, r: 0.0), simd_quatd(ix: 0.0, iy: 0.0, iz:-1.0, r: 0.0),
		
		//	2-fold rotations about edge centers
		simd_quatd(ix:-rhf, iy:-rhf, iz: 0.0, r: 0.0), simd_quatd(ix: rhf, iy: rhf, iz: 0.0, r: 0.0),
		simd_quatd(ix:-rhf, iy: rhf, iz: 0.0, r: 0.0), simd_quatd(ix: rhf, iy:-rhf, iz: 0.0, r: 0.0),
		simd_quatd(ix: 0.0, iy:-rhf, iz:-rhf, r: 0.0), simd_quatd(ix: 0.0, iy: rhf, iz: rhf, r: 0.0),
		simd_quatd(ix: 0.0, iy:-rhf, iz: rhf, r: 0.0), simd_quatd(ix: 0.0, iy: rhf, iz:-rhf, r: 0.0),
		simd_quatd(ix:-rhf, iy: 0.0, iz:-rhf, r: 0.0), simd_quatd(ix: rhf, iy: 0.0, iz: rhf, r: 0.0),
		simd_quatd(ix: rhf, iy: 0.0, iz:-rhf, r: 0.0), simd_quatd(ix:-rhf, iy: 0.0, iz: rhf, r: 0.0),
		
		//	3-fold rotations about vertices
		simd_quatd(ix:-0.5, iy:-0.5, iz:-0.5, r: 0.5), simd_quatd(ix: 0.5, iy: 0.5, iz: 0.5, r:-0.5),
		simd_quatd(ix:-0.5, iy:-0.5, iz: 0.5, r: 0.5), simd_quatd(ix: 0.5, iy: 0.5, iz:-0.5, r:-0.5),
		simd_quatd(ix:-0.5, iy: 0.5, iz:-0.5, r: 0.5), simd_quatd(ix: 0.5, iy:-0.5, iz: 0.5, r:-0.5),
		simd_quatd(ix:-0.5, iy: 0.5, iz: 0.5, r: 0.5), simd_quatd(ix: 0.5, iy:-0.5, iz:-0.5, r:-0.5),
		simd_quatd(ix: 0.5, iy:-0.5, iz:-0.5, r: 0.5), simd_quatd(ix:-0.5, iy: 0.5, iz: 0.5, r:-0.5),
		simd_quatd(ix: 0.5, iy:-0.5, iz: 0.5, r: 0.5), simd_quatd(ix:-0.5, iy: 0.5, iz:-0.5, r:-0.5),
		simd_quatd(ix: 0.5, iy: 0.5, iz:-0.5, r: 0.5), simd_quatd(ix:-0.5, iy:-0.5, iz: 0.5, r:-0.5),
		simd_quatd(ix: 0.5, iy: 0.5, iz: 0.5, r: 0.5), simd_quatd(ix:-0.5, iy:-0.5, iz:-0.5, r:-0.5),
		
		//	4-fold rotations about face centers
		simd_quatd(ix:-rhf, iy: 0.0, iz: 0.0, r: rhf), simd_quatd(ix: rhf, iy: 0.0, iz: 0.0, r:-rhf),
		simd_quatd(ix: rhf, iy: 0.0, iz: 0.0, r: rhf), simd_quatd(ix:-rhf, iy: 0.0, iz: 0.0, r:-rhf),
		simd_quatd(ix: 0.0, iy:-rhf, iz: 0.0, r: rhf), simd_quatd(ix: 0.0, iy: rhf, iz: 0.0, r:-rhf),
		simd_quatd(ix: 0.0, iy: rhf, iz: 0.0, r: rhf), simd_quatd(ix: 0.0, iy:-rhf, iz: 0.0, r:-rhf),
		simd_quatd(ix: 0.0, iy: 0.0, iz:-rhf, r: rhf), simd_quatd(ix: 0.0, iy: 0.0, iz: rhf, r:-rhf),
		simd_quatd(ix: 0.0, iy: 0.0, iz: rhf, r: rhf), simd_quatd(ix: 0.0, iy: 0.0, iz:-rhf, r:-rhf)
	]

	//	At the end of a rotation, how close must itsOrientation
	//	be to the nearest axis-aligned orientation in order
	//	to trigger a snap-to-axis?
	let theSnapToAxisTolerance = 0.999

	//	The given orientation will closely align
	//	with at most one of theAxisAlignedOrientations
	let theSnappedOrientation = theAxisAlignedOrientations.first(where: { alignedOrientation in
		dot(orientation.vector, alignedOrientation.vector) > theSnapToAxisTolerance })
	
	return theSnappedOrientation
}

// MARK: -
// MARK: Hit testing

//	When the user taps the display with his/her finger,
//	we'll need to decide whether it hits the slider.
//	Interpret the tap or click as a ray that begins at the user's eye,
//	passes through the selected point, and continues into the scene.
//
struct HitTestRay {

	//	Parameterize the ray as
	//
	//			p₀ + t(p₁ - p₀)
	//
	let p₀: SIMD3<Double>	//	in [0, n-1] maze coordinates
	let p₁: SIMD3<Double>	//	in [0, n-1] maze coordinates
}

func makeHitTestRay(
	touchPointIntrinsic: SIMD2<Double>,
	orientation: simd_quatd,
	n: Int	//	for n×n×n×n array of nodes
) -> HitTestRay {

	//	Initialize theRay in world coordinates.
	//
	//	makeProjectionMatrix() places the scene (d + boxSize) units
	//	in front of the user's eye, where d = perspectiveFactor * boxSize,
	//	so here we need to undo that motion to get back into world coordinates.
	//
	var p₀ = SIMD3<Double>(0.0, 0.0, -(perspectiveFactor + 1.0) * boxSize)		//	user's eye
	var p₁ = SIMD3<Double>(touchPointIntrinsic[0], touchPointIntrinsic[1], -boxSize)
	
	//	Undo the rotational part of the view matrix,
	//	that is, of theViewMatrixR in prepareUniformBuffer().
	//
	p₀ = orientation.inverse.act(p₀)
	p₁ = orientation.inverse.act(p₁)

	//	Undo the dilational and translational parts of the view matrix,
	//	that is, of theViewMatrixDT in prepareUniformBuffer(), which acts as
	//
	//		xyz -> -1.0 + 2.0 * ( xyz / (n - 1) )
	//
	p₀ = (p₀ + SIMD3<Double>(1.0, 1.0, 1.0)) * Double(0.5) * Double(n - 1)
	p₁ = (p₁ + SIMD3<Double>(1.0, 1.0, 1.0)) * Double(0.5) * Double(n - 1)
	
	return HitTestRay(p₀: p₀, p₁: p₁)
}

func rayHitsSlider(
	ray: HitTestRay,
	sliderCenter: SIMD3<Double>,
	haloRadius: Double
) -> Bool {
	
	return rayIntersectsSphere(
		ray: ray,
		sphereCenter: sliderCenter,
		sphereRadius: haloRadius)
}

func rayIntersectsSphere(
	ray: HitTestRay,
	sphereCenter: SIMD3<Double>,
	sphereRadius: Double
) -> Bool {

	//	For brevity…
	let p₀ = ray.p₀
	let p₁ = ray.p₁
	let q = sphereCenter
	let r = sphereRadius
	
	//	Let
	//		p(t) = p₀ + t(p₁ - p₀)
	//
	//	We seek a value of t for which
	//
	//		| p(t) - q |²
	//
	//	is a minimum.  Rewriting that expression as a dot product gives
	//
	//		  (p(t) - q)·(p(t) - q)
	//		= p(t)·p(t) - 2 p(t)·q + q·q
	//		= (p₀ + t(p₁ - p₀))·(p₀ + t(p₁ - p₀)) - 2(p₀ + t(p₁ - p₀))·q + q·q
	//		= p₀·p₀ + 2p₀·(p₁ - p₀)t + (p₁ - p₀)·(p₁ - p₀)t² - 2p₀·q - 2(p₁ - p₀)·qt + q·q
	//		=  (p₁ - p₀)·(p₁ - p₀) t²
	//		  + 2 (p₀ - q)·(p₁ - p₀) t
	//		  + (p₀ - q)·(p₀ - q)
	//
	//	To find the value of t for which that expression reaches its minimum,
	//	take the derivative and set it equal to zero
	//
	//		2 (p₁ - p₀)·(p₁ - p₀) t + 2 (p₀ - q)·(p₁ - p₀) = 0
	//
	//	We can rewrite that as
	//
	//		a t + b = 0
	//
	//	where
	//
	//		a = (p₁ - p₀)·(p₁ - p₀)
	//		b = (p₀ - q)·(p₁ - p₀)
	//
	let a = dot(p₁ - p₀, p₁ - p₀)
	let b = dot(p₀ - q, p₁ - p₀)
	if a <= 0.0 {
		assertionFailure("Internal error:  p₀ and p₁ coincide in rayIntersectsSphere()")
		return false
	}
	let t = -b / a
	let pt = p₀ + t*(p₁ - p₀)
	let theDistanceSquared = dot(pt - q, pt - q)	//	at point of closest approach

	return theDistanceSquared <= r*r	//	Does the ray intersect the sphere?
}
